<?php
include_once '../config/header.php';
include_once '../config/helper.php';
include_once '../config/database.php';
include_once '../objects/vecinos.php';
include_once '../token/validatetoken.php';

// Inicializar la conexión a la base de datos
$database = new Database();
$db = $database->getConnection();

// Obtener datos de la solicitud
$data = json_decode(file_get_contents("php://input"));

// Validar datos necesarios
if (!isset($data->latitud) || !isset($data->longitud)) {
    http_response_code(400);
    echo json_encode(array(
        "status" => "error",
        "code" => 0,
        "message" => "Faltan datos requeridos (latitud, longitud).",
        "document" => ""
    ));
    exit();
}

// Radio de búsqueda predeterminado (3 km)
$radio = isset($data->radio) ? $data->radio : 3;

// Minutos de inactividad máximos para considerar una ubicación actualizada
$maxInactiveMinutes = isset($data->max_inactive_minutos) ? $data->max_inactive_minutos : 15;

// Definir tiempo máximo de inactividad
$maxInactiveTimestamp = date('Y-m-d H:i:s', strtotime("-$maxInactiveMinutes minutes"));

try {
    // Consulta priorizando ubicación en tiempo real
    $query = "SELECT v.id, v.nombre, v.apellido, v.telefono, v.push_token, 
            COALESCE(vu.latitud, v.latitud) as latitud, 
            COALESCE(vu.longitud, v.longitud) as longitud, 
            vu.timestamp,
            CASE 
                WHEN vu.id IS NOT NULL AND vu.timestamp > :max_inactive_timestamp THEN 1 
                ELSE 0 
            END as ubicacion_tiempo_real,
            vu.timestamp as ultima_actividad,
            (
                6371 * acos(
                    cos(radians(:latitude)) * cos(radians(COALESCE(vu.latitud, v.latitud))) *
                    cos(radians(COALESCE(vu.longitud, v.longitud)) - radians(:longitude)) +
                    sin(radians(:latitude)) * sin(radians(COALESCE(vu.latitud, v.latitud)))
                )
            ) AS distancia
        FROM vecinos v
        LEFT JOIN (
            SELECT vu1.* 
            FROM vecinos_ubicacion vu1
            INNER JOIN (
                SELECT id_vecino, MAX(timestamp) as max_timestamp 
                FROM vecinos_ubicacion 
                GROUP BY id_vecino
            ) vu2 
            ON vu1.id_vecino = vu2.id_vecino AND vu1.timestamp = vu2.max_timestamp
        ) vu ON v.id = vu.id_vecino
        WHERE v.vigilante = 1 AND v.activo = 1
        HAVING distancia <= :radius
        ORDER BY ubicacion_tiempo_real DESC, distancia ASC";
        
    $stmt = $db->prepare($query);
    $stmt->bindParam(':max_inactive_timestamp', $maxInactiveTimestamp);
    $stmt->bindParam(':latitude', $data->latitud);
    $stmt->bindParam(':longitude', $data->longitud);
    $stmt->bindParam(':radius', $radio);
    $stmt->execute();

    $num = $stmt->rowCount();

    if ($num > 0) {
        $voluntarios_arr = array();
        $voluntarios_arr["records"] = array();
        $voluntarios_arr["count"] = $num;

        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            extract($row);

            $voluntario_item = array(
                "id" => $id,
                "nombre" => $nombre,
                "apellido" => $apellido,
                "telefono" => $telefono,
                "push_token" => $push_token,
                "latitud" => $latitud,
                "longitud" => $longitud,
                "distancia" => round($distancia, 2),
                "ubicacion_tiempo_real" => $ubicacion_tiempo_real,
                "ultima_actividad" => $timestamp
            );

            array_push($voluntarios_arr["records"], $voluntario_item);
        }

        http_response_code(200);
        echo json_encode($voluntarios_arr);
    } else {
        http_response_code(200);
        echo json_encode(array(
            "status" => "success",
            "code" => 1,
            "message" => "No se encontraron voluntarios cercanos en un radio de $radio km.",
            "records" => array(),
            "count" => 0
        ));
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(array(
        "status" => "error",
        "code" => 0,
        "message" => "Error al buscar voluntarios: " . $e->getMessage(),
        "document" => ""
    ));
}
?>