<?php
include_once '../config/header.php';
include_once '../config/helper.php';
include_once '../config/database.php';
include_once '../objects/vecinos.php';
include_once '../token/validatetoken.php';

// Crear directorio de logs si no existe
if (!file_exists('../logs')) {
    mkdir('../logs', 0777, true);
}

// Crear un archivo de log para depuración
$logFile = "../logs/location_update_" . date('Y-m-d') . ".log";
file_put_contents($logFile, date('Y-m-d H:i:s') . " - Solicitud recibida\n", FILE_APPEND);

// Inicializar la conexion a la base de datos
$database = new Database();
$db = $database->getConnection();
file_put_contents($logFile, date('Y-m-d H:i:s') . " - Conexión a base de datos establecida\n", FILE_APPEND);

// Obtener datos de la solicitud
$inputData = file_get_contents("php://input");
file_put_contents($logFile, date('Y-m-d H:i:s') . " - Datos recibidos: " . $inputData . "\n", FILE_APPEND);

$data = json_decode($inputData);

// Validar datos necesarios
if(!isset($data->id) || !isset($data->latitud) || !isset($data->longitud)){
    http_response_code(400);
    $errorResponse = array(
        "status" => "error",
        "code" => 0,
        "message" => "Faltan datos requeridos (id, latitud, longitud).",
        "document" => ""
    );
    echo json_encode($errorResponse);
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error 400: Faltan datos necesarios\n", FILE_APPEND);
    exit();
}

try {
    // Verificar si el usuario es voluntario
    $esVoluntario = false;
    
    $query = "SELECT vigilante FROM vecinos WHERE id = :id";
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Query para verificar si es voluntario: " . $query . "\n", FILE_APPEND);
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $data->id);
    $stmt->execute();
    
    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $esVoluntario = (int)$row['vigilante'] === 1;
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Usuario ID: {$data->id}, Es voluntario: " . ($esVoluntario ? "Sí" : "No") . "\n", FILE_APPEND);
    } else {
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Usuario ID: {$data->id} no encontrado\n", FILE_APPEND);
    }
    
    // Actualizar la ubicacion en la tabla principal
    $query = "UPDATE vecinos SET 
                latitud = :latitud, 
                longitud = :longitud, 
                ultima_actualizacion = NOW()
              WHERE id = :id";
    
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Query para actualizar ubicación principal: " . $query . "\n", FILE_APPEND);
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $data->id);
    $stmt->bindParam(':latitud', $data->latitud);
    $stmt->bindParam(':longitud', $data->longitud);
    
    $updateResult = $stmt->execute();
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Actualización tabla principal: " . ($updateResult ? "Éxito" : "Fallo") . "\n", FILE_APPEND);
    
    if($updateResult) {
        // Si es voluntario, guardar tambien en la tabla de ubicaciones historicas
        $realTimeSaved = false;
        
        if($esVoluntario || (isset($data->es_voluntario) && $data->es_voluntario == 1)) {
            $precision = isset($data->precision) ? $data->precision : 'normal';
            
            // Verificar si la tabla existe
            try {
                $tableExistsQuery = "SHOW TABLES LIKE 'vecinos_ubicacion'";
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Verificando si existe la tabla: " . $tableExistsQuery . "\n", FILE_APPEND);
                
                $tableResult = $db->query($tableExistsQuery);
                $tableExists = $tableResult && $tableResult->rowCount() > 0;
                
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Tabla vecinos_ubicacion existe: " . ($tableExists ? "Sí" : "No") . "\n", FILE_APPEND);
            } catch(Exception $tableEx) {
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error al verificar tabla: " . $tableEx->getMessage() . "\n", FILE_APPEND);
                $tableExists = false;
            }
            
            // Verificar y crear tabla si es necesario
            if (!$tableExists) {
                try {
                    $createTableQuery = "CREATE TABLE IF NOT EXISTS vecinos_ubicacion (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        id_vecino INT NOT NULL,
                        latitud DECIMAL(10,8) NOT NULL,
                        longitud DECIMAL(11,8) NOT NULL,
                        precision VARCHAR(20) DEFAULT 'normal',
                        timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        INDEX (id_vecino),
                        INDEX (timestamp)
                    )";
                    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Intentando crear tabla: " . $createTableQuery . "\n", FILE_APPEND);
                    
                    $createResult = $db->exec($createTableQuery);
                    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Creación de tabla: " . ($createResult !== false ? "Éxito" : "Fallo") . "\n", FILE_APPEND);
                } catch(Exception $createEx) {
                    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error al crear tabla: " . $createEx->getMessage() . "\n", FILE_APPEND);
                }
            }
            
            // Ahora intento de inserción en la tabla de ubicaciones
            try {
                $query = "INSERT INTO vecinos_ubicacion (id_vecino, latitud, longitud, precision) 
                        VALUES (:id_vecino, :latitud, :longitud, :precision)";
                
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Query para guardar ubicación en tiempo real: " . $query . "\n", FILE_APPEND);
                
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id_vecino', $data->id);
                $stmt->bindParam(':latitud', $data->latitud);
                $stmt->bindParam(':longitud', $data->longitud);
                $stmt->bindParam(':precision', $precision);
                
                $insertResult = $stmt->execute();
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Inserción en tabla ubicación: " . ($insertResult ? "Éxito" : "Fallo") . "\n", FILE_APPEND);
                
                if($insertResult) {
                    $realTimeSaved = true;
                    
                    // Limitar historico a las ultimas 20 ubicaciones por usuario
                    try {
                        $query = "DELETE FROM vecinos_ubicacion 
                                WHERE id_vecino = :id_vecino 
                                AND id NOT IN (
                                    SELECT id FROM (
                                        SELECT id FROM vecinos_ubicacion 
                                        WHERE id_vecino = :id_vecino 
                                        ORDER BY timestamp DESC 
                                        LIMIT 20
                                    ) temp
                                )";
                        
                        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Query para limpiar ubicaciones antiguas: " . $query . "\n", FILE_APPEND);
                        
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':id_vecino', $data->id);
                        $cleanupResult = $stmt->execute();
                        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Limpieza de registros antiguos: " . ($cleanupResult ? "Éxito" : "Fallo") . "\n", FILE_APPEND);
                    } catch(Exception $cleanupEx) {
                        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error en limpieza: " . $cleanupEx->getMessage() . "\n", FILE_APPEND);
                    }
                }
            } catch(Exception $insertEx) {
                file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error en inserción: " . $insertEx->getMessage() . "\n", FILE_APPEND);
            }
        }
        
        http_response_code(200);
        $successResponse = array(
            "status" => "success",
            "code" => 1,
            "message" => "Ubicacion actualizada correctamente." . 
                        ($realTimeSaved ? " Historial de ubicaciones actualizado." : ""),
            "document" => "",
            "es_voluntario" => $esVoluntario
        );
        echo json_encode($successResponse);
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Respuesta exitosa enviada\n", FILE_APPEND);
    } else {
        http_response_code(500);
        $errorResponse = array(
            "status" => "error",
            "code" => 0,
            "message" => "No se pudo actualizar la ubicacion.",
            "document" => ""
        );
        echo json_encode($errorResponse);
        file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error 500: No se pudo actualizar la ubicación\n", FILE_APPEND);
    }
} catch(PDOException $e) {
    // Manejar errores específicos de PDO/Base de datos
    http_response_code(500);
    $errorResponse = array(
        "status" => "error",
        "code" => 0,
        "message" => "Error de base de datos: " . $e->getMessage(),
        "document" => ""
    );
    echo json_encode($errorResponse);
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error PDO: " . $e->getMessage() . "\n", FILE_APPEND);
} catch(Exception $e) {
    // Manejar otros errores
    http_response_code(500);
    $errorResponse = array(
        "status" => "error",
        "code" => 0,
        "message" => "Error general al actualizar la ubicacion: " . $e->getMessage(),
        "document" => ""
    );
    echo json_encode($errorResponse);
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Error general: " . $e->getMessage() . "\n", FILE_APPEND);
}
?>